<?php
/**
 * クリエイターデータ取得API
 * noteクリエイター向けダッシュボード
 * 
 * note API v2の公開エンドポイントを使用して、クリエイターの公開情報を取得
 */

require_once __DIR__ . '/../includes/auth.php';

// 認証チェック - 未ログインの場合はエラーを返す
if (!isLoggedIn()) {
    http_response_code(401);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['error' => '認証が必要です。ログインしてください。']);
    exit;
}

// JSONレスポンスヘッダー
header('Content-Type: application/json; charset=utf-8');

// POSTリクエストのみ許可
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'POSTメソッドのみ許可されています。']);
    exit;
}

// リクエストボディを取得
$input = json_decode(file_get_contents('php://input'), true);

// CSRFトークン検証
$csrfToken = $input['csrf_token'] ?? '';
if (!validateCsrfToken($csrfToken)) {
    http_response_code(403);
    echo json_encode(['error' => '不正なリクエストです。']);
    exit;
}

// クリエイターIDを取得
$creatorId = trim($input['creator_id'] ?? '');

if (empty($creatorId)) {
    http_response_code(400);
    echo json_encode(['error' => 'クリエイターIDを入力してください。']);
    exit;
}

// クリエイターIDのバリデーション（英数字とアンダースコアのみ）
if (!preg_match('/^[a-zA-Z0-9_]+$/', $creatorId)) {
    http_response_code(400);
    echo json_encode(['error' => 'クリエイターIDの形式が正しくありません。']);
    exit;
}

try {
    // note API v2からクリエイター情報を取得
    $creatorData = fetchCreatorInfo($creatorId);
    
    if ($creatorData === null) {
        http_response_code(404);
        echo json_encode(['error' => '指定されたクリエイターが見つかりませんでした。']);
        exit;
    }
    
    // 記事一覧を取得
    $articles = fetchCreatorArticles($creatorId);
    
    // 統計情報を計算
    $stats = calculateStats($articles);
    
    // レスポンスを返す
    echo json_encode([
        'success' => true,
        'creator' => $creatorData,
        'articles' => $articles,
        'stats' => $stats
    ], JSON_UNESCAPED_UNICODE);
    
} catch (Exception $e) {
    error_log('API Error: ' . $e->getMessage());
    http_response_code(500);
    echo json_encode(['error' => 'データの取得中にエラーが発生しました。']);
}

/**
 * note APIからクリエイター情報を取得
 * 
 * @param string $creatorId クリエイターID
 * @return array|null クリエイター情報
 */
function fetchCreatorInfo(string $creatorId): ?array
{
    $url = "https://note.com/api/v2/creators/{$creatorId}";
    
    $response = makeApiRequest($url);
    
    if ($response === null || !isset($response['data'])) {
        return null;
    }
    
    $data = $response['data'];
    
    return [
        'id' => $data['id'] ?? null,
        'urlname' => $data['urlname'] ?? $creatorId,
        'name' => $data['name'] ?? $data['nickname'] ?? $creatorId,
        'description' => $data['description'] ?? '',
        'avatar_url' => $data['user_profile_image_path'] ?? $data['profile_image_path'] ?? '',
        'follower_count' => $data['follower_count'] ?? 0,
        'following_count' => $data['following_count'] ?? 0,
        'note_count' => $data['note_count'] ?? 0
    ];
}

/**
 * note APIからクリエイターの記事一覧を取得
 * 
 * @param string $creatorId クリエイターID
 * @param int $page ページ番号
 * @return array 記事一覧
 */
function fetchCreatorArticles(string $creatorId, int $page = 1): array
{
    $articles = [];
    $maxPages = 5; // 最大5ページまで取得
    
    for ($p = 1; $p <= $maxPages; $p++) {
        $url = "https://note.com/api/v2/creators/{$creatorId}/contents?kind=note&page={$p}";
        
        $response = makeApiRequest($url);
        
        if ($response === null || !isset($response['data']['contents'])) {
            break;
        }
        
        $contents = $response['data']['contents'];
        
        if (empty($contents)) {
            break;
        }
        
        foreach ($contents as $content) {
            $articles[] = [
                'id' => $content['id'] ?? null,
                'title' => $content['name'] ?? '',
                'like_count' => $content['like_count'] ?? 0,
                'published_at' => $content['published_at'] ?? $content['created_at'] ?? '',
                'url' => "https://note.com/{$creatorId}/n/" . ($content['key'] ?? ''),
                'is_premium' => $content['price'] > 0 ?? false
            ];
        }
        
        // 次のページがない場合は終了
        if (!isset($response['data']['is_last_page']) || $response['data']['is_last_page']) {
            break;
        }
    }
    
    return $articles;
}

/**
 * 統計情報を計算
 * 
 * @param array $articles 記事一覧
 * @return array 統計情報
 */
function calculateStats(array $articles): array
{
    $totalArticles = count($articles);
    $totalLikes = 0;
    
    foreach ($articles as $article) {
        $totalLikes += $article['like_count'];
    }
    
    $avgLikes = $totalArticles > 0 ? round($totalLikes / $totalArticles, 1) : 0;
    
    return [
        'total_articles' => $totalArticles,
        'total_likes' => $totalLikes,
        'avg_likes' => $avgLikes
    ];
}

/**
 * APIリクエストを実行
 * 
 * @param string $url リクエストURL
 * @return array|null レスポンスデータ
 */
function makeApiRequest(string $url): ?array
{
    $options = [
        'http' => [
            'method' => 'GET',
            'header' => [
                'User-Agent: NoteCreatorDashboard/1.0',
                'Accept: application/json'
            ],
            'timeout' => 10
        ]
    ];
    
    $context = stream_context_create($options);
    
    $response = @file_get_contents($url, false, $context);
    
    if ($response === false) {
        return null;
    }
    
    return json_decode($response, true);
}
