<?php
/**
 * 認証ヘルパー関数
 * noteクリエイター向けダッシュボード
 */

require_once __DIR__ . '/../config/database.php';

// セッション設定
ini_set('session.cookie_httponly', 1);
ini_set('session.use_only_cookies', 1);
ini_set('session.cookie_secure', isset($_SERVER['HTTPS']) ? 1 : 0);

/**
 * セッションを開始（まだ開始されていない場合）
 */
function startSecureSession(): void
{
    if (session_status() === PHP_SESSION_NONE) {
        session_start();
    }
}

/**
 * ユーザーがログイン済みかチェック
 * 
 * @return bool ログイン済みの場合true
 */
function isLoggedIn(): bool
{
    startSecureSession();
    return isset($_SESSION['user_id']) && isset($_SESSION['logged_in']) && $_SESSION['logged_in'] === true;
}

/**
 * ログインを要求（未ログインの場合リダイレクト）
 * 
 * @param string $redirectUrl リダイレクト先URL
 */
function requireLogin(string $redirectUrl = 'login.php'): void
{
    if (!isLoggedIn()) {
        header('Location: ' . $redirectUrl);
        exit;
    }
}

/**
 * ユーザー認証を実行
 * 
 * @param string $username ユーザー名
 * @param string $password パスワード
 * @return array|false 成功時はユーザー情報、失敗時はfalse
 */
function authenticateUser(string $username, string $password): array|false
{
    $pdo = getDbConnection();
    if ($pdo === null) {
        return false;
    }
    
    try {
        $stmt = $pdo->prepare('SELECT id, username, password_hash, email, is_active FROM users WHERE username = ? AND is_active = 1');
        $stmt->execute([$username]);
        $user = $stmt->fetch();
        
        if ($user && password_verify($password, $user['password_hash'])) {
            // 最終ログイン日時を更新
            $updateStmt = $pdo->prepare('UPDATE users SET last_login = NOW() WHERE id = ?');
            $updateStmt->execute([$user['id']]);
            
            // ログイン履歴を記録
            logLoginAttempt($user['id'], $username, 'success');
            
            return $user;
        }
        
        // ログイン失敗を記録
        logLoginAttempt(null, $username, 'failed', 'Invalid credentials');
        return false;
    } catch (PDOException $e) {
        error_log('Authentication error: ' . $e->getMessage());
        return false;
    }
}

/**
 * ログイン処理を実行
 * 
 * @param array $user ユーザー情報
 */
function loginUser(array $user): void
{
    startSecureSession();
    
    // セッションIDを再生成（セッション固定攻撃対策）
    session_regenerate_id(true);
    
    $_SESSION['user_id'] = $user['id'];
    $_SESSION['username'] = $user['username'];
    $_SESSION['logged_in'] = true;
    $_SESSION['login_time'] = time();
    $_SESSION['ip_address'] = $_SERVER['REMOTE_ADDR'] ?? '';
    $_SESSION['user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? '';
}

/**
 * ログアウト処理を実行
 */
function logoutUser(): void
{
    startSecureSession();
    
    // セッション変数をクリア
    $_SESSION = [];
    
    // セッションクッキーを削除
    if (ini_get('session.use_cookies')) {
        $params = session_get_cookie_params();
        setcookie(
            session_name(),
            '',
            time() - 42000,
            $params['path'],
            $params['domain'],
            $params['secure'],
            $params['httponly']
        );
    }
    
    // セッションを破棄
    session_destroy();
}

/**
 * ログイン試行を記録
 * 
 * @param int|null $userId ユーザーID
 * @param string $username ユーザー名
 * @param string $status ステータス（success/failed）
 * @param string|null $failureReason 失敗理由
 */
function logLoginAttempt(?int $userId, string $username, string $status, ?string $failureReason = null): void
{
    $pdo = getDbConnection();
    if ($pdo === null) {
        return;
    }
    
    try {
        $stmt = $pdo->prepare('INSERT INTO login_history (user_id, username, ip_address, user_agent, login_status, failure_reason) VALUES (?, ?, ?, ?, ?, ?)');
        $stmt->execute([
            $userId,
            $username,
            $_SERVER['REMOTE_ADDR'] ?? '',
            $_SERVER['HTTP_USER_AGENT'] ?? '',
            $status,
            $failureReason
        ]);
    } catch (PDOException $e) {
        error_log('Failed to log login attempt: ' . $e->getMessage());
    }
}

/**
 * CSRFトークンを生成
 * 
 * @return string CSRFトークン
 */
function generateCsrfToken(): string
{
    startSecureSession();
    
    if (empty($_SESSION['csrf_token'])) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
    }
    
    return $_SESSION['csrf_token'];
}

/**
 * CSRFトークンを検証
 * 
 * @param string $token 検証するトークン
 * @return bool 有効な場合true
 */
function validateCsrfToken(string $token): bool
{
    startSecureSession();
    
    if (empty($_SESSION['csrf_token'])) {
        return false;
    }
    
    return hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * 新規ユーザーを作成
 * 
 * @param string $username ユーザー名
 * @param string $password パスワード
 * @param string|null $email メールアドレス
 * @return int|false 成功時はユーザーID、失敗時はfalse
 */
function createUser(string $username, string $password, ?string $email = null): int|false
{
    $pdo = getDbConnection();
    if ($pdo === null) {
        return false;
    }
    
    try {
        $passwordHash = password_hash($password, PASSWORD_DEFAULT);
        
        $stmt = $pdo->prepare('INSERT INTO users (username, password_hash, email) VALUES (?, ?, ?)');
        $stmt->execute([$username, $passwordHash, $email]);
        
        return (int)$pdo->lastInsertId();
    } catch (PDOException $e) {
        error_log('Failed to create user: ' . $e->getMessage());
        return false;
    }
}

/**
 * 現在ログイン中のユーザー情報を取得
 * 
 * @return array|null ユーザー情報
 */
function getCurrentUser(): ?array
{
    if (!isLoggedIn()) {
        return null;
    }
    
    return [
        'id' => $_SESSION['user_id'],
        'username' => $_SESSION['username']
    ];
}
