<?php
/**
 * 初期ユーザー作成スクリプト
 * noteクリエイター向けダッシュボード
 * 
 * 使用方法: php setup/init_user.php <username> <password> [email]
 */

require_once __DIR__ . '/../config/database.php';

// コマンドライン引数をチェック
if ($argc < 3) {
    echo "使用方法: php setup/init_user.php <username> <password> [email]\n";
    echo "例: php setup/init_user.php admin mypassword admin@example.com\n";
    exit(1);
}

$username = $argv[1];
$password = $argv[2];
$email = $argv[3] ?? null;

// バリデーション
if (strlen($username) < 3 || strlen($username) > 50) {
    echo "エラー: ユーザー名は3〜50文字で入力してください。\n";
    exit(1);
}

if (strlen($password) < 6) {
    echo "エラー: パスワードは6文字以上で入力してください。\n";
    exit(1);
}

// データベース接続
$pdo = getDbConnection();
if ($pdo === null) {
    echo "エラー: データベースに接続できませんでした。\n";
    echo "config/database.php の設定を確認してください。\n";
    exit(1);
}

try {
    // ユーザーが既に存在するかチェック
    $stmt = $pdo->prepare('SELECT id FROM users WHERE username = ?');
    $stmt->execute([$username]);
    
    if ($stmt->fetch()) {
        echo "エラー: ユーザー名 '{$username}' は既に使用されています。\n";
        exit(1);
    }
    
    // パスワードをハッシュ化
    $passwordHash = password_hash($password, PASSWORD_DEFAULT);
    
    // ユーザーを作成
    $stmt = $pdo->prepare('INSERT INTO users (username, password_hash, email) VALUES (?, ?, ?)');
    $stmt->execute([$username, $passwordHash, $email]);
    
    $userId = $pdo->lastInsertId();
    
    echo "ユーザーを作成しました。\n";
    echo "  ID: {$userId}\n";
    echo "  ユーザー名: {$username}\n";
    if ($email) {
        echo "  メール: {$email}\n";
    }
    echo "\nログインページからログインできます。\n";
    
} catch (PDOException $e) {
    echo "エラー: ユーザーの作成に失敗しました。\n";
    echo "詳細: " . $e->getMessage() . "\n";
    exit(1);
}
